"""
This file is part of blender-osm (OpenStreetMap importer for Blender).
Copyright (C) 2014-2018 Vladimir Elistratov
prokitektura+support@gmail.com

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.

Modified by Wiek Schoenmakers for Condor 3 <http://www.condorsoaring.com>
"""

import math, bpy

# see conversion formulas at
# http://en.wikipedia.org/wiki/Transverse_Mercator_projection
# and
# http://mathworld.wolfram.com/MercatorProjection.html
class TransverseMercator:
    radius = 6378137.

    def __init__(self, **kwargs):
        # setting default values
        self.lat = 0. # in degrees
        self.lon = 0. # in degrees
        self.k = 1. # scale factor
        
        for attr in kwargs:
            setattr(self, attr, kwargs[attr])
        self.latInRadians = math.radians(self.lat)

    def fromGeographic(self, lat, lon):
        PI = 3.14159265
        FOURTHPI = PI / 4
        deg2rad = PI / 180.0
        rad2deg = 180.0 / PI
        EquatorialRadius = 6378137
        eccSquared = 0.00669438
        ZoneNumber = bpy.context.scene['ZoneNumber']
        TranslateX = bpy.context.scene['TranslateX']
        TranslateY = bpy.context.scene['TranslateY']

        k0 = 0.9996
        LongTemp = (lon+180)-int((lon+180) / 360) * 360 - 180
        LatRad = lat*deg2rad
        LongRad = LongTemp*deg2rad

        LongOrigin = (ZoneNumber - 1) * 6 -180 + 3 # +3 puts origin in middle of zone
        LongOriginRad = LongOrigin * deg2rad

        eccPrimeSquared = (eccSquared)/(1-eccSquared)

        N = EquatorialRadius/math.sqrt(1-eccSquared*math.sin(LatRad)*math.sin(LatRad))
        T = math.tan(LatRad)*math.tan(LatRad)
        C = eccPrimeSquared*math.cos(LatRad)*math.cos(LatRad)
        A = math.cos(LatRad)*(LongRad-LongOriginRad)

        M = EquatorialRadius*((1 - eccSquared/4 - 3*eccSquared*eccSquared/64 - 5*eccSquared*eccSquared*eccSquared/256)*LatRad - (3*eccSquared/8 + 3*eccSquared*eccSquared/32 + 45*eccSquared*eccSquared*eccSquared/1024)*math.sin(2*LatRad) + (15*eccSquared*eccSquared/256 + 45*eccSquared*eccSquared*eccSquared/1024)*math.sin(4*LatRad) - (35*eccSquared*eccSquared*eccSquared/3072)*math.sin(6*LatRad))

        x = (k0*N*(A+(1-T+C)*A*A*A/6 + (5-18*T+T*T+72*C-58*eccPrimeSquared)*A*A*A*A*A/120) + 500000.0)

        y = (k0*(M+N*math.tan(LatRad)*(A*A/2+(5-T+9*C+4*C*C)*A*A*A*A/24 + (61-58*T+T*T+600*C-330*eccPrimeSquared)*A*A*A*A*A*A/720)))

        if (self.lat < 0):
            y = y + 10000000 # 10000000 meter offset for southern hemisphere
        return (x + TranslateX, y + TranslateY, 0.0)

    def toGeographic(self, x, y):
        x = x/(self.k * self.radius)
        y = y/(self.k * self.radius)
        D = y + self.latInRadians
        lon = math.atan(math.sinh(x)/math.cos(D))
        lat = math.asin(math.sin(D)/math.cosh(x))

        lon = self.lon + math.degrees(lon)
        lat = math.degrees(lat)
        return (lat, lon)
        