import math
from util.osm import parseNumber
from .scope import *


"""
Code to generate dict with CSS colors from a tuple like

colors = (
    ("aliceblue", "#F0F8FF"),
    ...
    ("yellowgreen", "#9ACD32")
)

for c in colors:
    v = list(map(lambda component: component/255, bytes.fromhex(c[1][1:])))
    print("    \"{0}\": ({1:g}, {2:g}, {3:g}, 1.),".format(c[0],round(v[0],3), round(v[1],3), round(v[2],3)))
"""

colors = {
    "aliceblue": (1, 1, 1, 1),
    
}


# <hexdigits> are used to check validity of colors in the hex form
_hexdigits = set("0123456789abcdef")


def getColor(color):
    """
    Returns Python tuple of 3 float values between 0. and 1. from a string,
    which can be either a hex or a CSS color
    """
    return colors[color] if color in colors else getColorFromHex(color)


def normalizeColor(color):
    """
    Check the validity of the Python string <color> for a color and
    returns the string in the lower case if it's valid or None otherwise.
    If the string is hex color, the resulting string is composed of exactly 6 character
    without leading character like <#>.
    """
    if color is None:
        return None
    color = color.lower()
    if not color in colors:
        numCharacters = len(color)
        if numCharacters == 7:
            color = color[1:]
        elif numCharacters in (3,4):
            # <color> has the form like <fff> or <#fff>
            color = "".join( 2*letter for letter in (color[-3:] if numCharacters==4 else color) )
        elif numCharacters != 6:
            # invalid
            return None
        # check that all characters in <color> are valid for a hex string
        if not all(c in _hexdigits for c in color):
            # invalid
            return None
    return color


def getColorFromHex(color):
    return tuple( c/255. for c in bytes.fromhex("%sff" % color) )


class Value:
    
    def __init__(self, value):
        self.value = value
        

class _Value:
    """
    The base class for the classes below
    """
    
    id = 0
    
    def __init__(self):
        # <self.id> is used to store the value in a style block cache
        self.id = _Value.getId()
        # the scope is per item by default
        self.scope = None
    
    @staticmethod
    def getId():
        _Value.id += 1
        return _Value.id

    def getValue(self, item):
        if self.scope:
            cache = item.getCache(self.scope)
            if self.id in cache:
                value = cache[self.id]
            else:
                value = self._getValue(item)
                # keep the value in the cache
                cache[self.id] = value
        else:
            value = self._getValue(item)
        return value


class Alternatives(_Value):
    """
    A value out of two or more alternatives
    """
    def __init__(self, *values):
        super().__init__()
        for value in values:
            if isinstance(value, Scope):
                value.value.scope = value.scope
        self.values = tuple(value.value if isinstance(value, Scope) else value for value in values)
    
    def _getValue(self, item):
        for value in self.values:
            value = value.getValue(item)
            if not value is None:
                break
        return value


class Constant:
    
    def __init__(self, value):
        self._value = value
    
    def getValue(self, item):
        return self._value


class FromAttr(_Value):
    """
    Take the value from the footprint data attribute
    """
    Integer = 1
    Float = 2
    String = 3
    Color = 4
    
    Positive = 4
    NonNegative = 5
    
    def __init__(self, attr, valueType, valueCondition=None):
        super().__init__()
        self.attr = attr
        self.valueType = valueType
        self.valueCondition = valueCondition
    
    def _getAttrValue(self, item):
        return (item.footprint if item.footprint else item).attr(self.attr)
    
    def _getValue(self, item):
        value = self._getAttrValue(item)
        if value is None:
            return None
        valueCondition = self.valueCondition
        if self.valueType is FromAttr.Integer:
            if not value is None:
                value = parseNumber(value)
                if not value is None:
                    value = math.ceil(value)
                    if valueCondition is FromAttr.Positive and value < 1.:
                        value = None
                    elif valueCondition is FromAttr.NonNegative and value < 0.:
                        value = None
        elif self.valueType is FromAttr.Float:
            if not value is None:
                value = parseNumber(value)
                if not value is None:
                    if valueCondition is FromAttr.Positive and value <= 0.:
                        value = None
                    elif valueCondition is FromAttr.NonNegative and value < 0.:
                        value = None
        elif self.valueType is FromAttr.Color:
            if isinstance(value, str):
                value = normalizeColor(value)
                value = getColor(value) if value else None
            else:
                value = None
        elif valueCondition:
            # We have a string and <valueCondition> is dictionary
            # where the keys are the possible values for the string
            if not value in valueCondition:
                value = None
        return value


class FromBldgAttr(FromAttr):
    
    def _getAttrValue(self, item):
        return item.building.attr(self.attr)


class Conditional:
    """
    Return the supplied value if the condition is True or None
    """
    def __init__(self, condition, value):
        if isinstance(value, Scope):
            value.value.scope = value.scope
            value = value.value
        self._value = value
        self.condition = condition
    
    def getValue(self, item):
        return self._value.getValue(item) if self.condition(item) else None


class FromStyleBlockAttr:
    # The folowing static variables define the possible values for <itemType>, i.e.
    # from which style block the given style block attribute should be taken
    Footprint = 1
    Self = 2
    Parent = 3
    
    def __init__(self, attr, itemType):
        self.attr = attr
        self.itemType = itemType
    
    def getValue(self, item):
        itemType = self.itemType
        
        if itemType is FromStyleBlockAttr.Footprint:
            item = item.footprint
        elif itemType is FromStyleBlockAttr.Parent:
            item = item.parent
        
        return item.getStyleBlockAttr(self.attr)


from util.random import RandomNormal as RandomNormalBase
class RandomNormal(_Value):
    """
    A wrapper for <util.random.RandomNormal>
    """
    def __init__(self, mean):
        super().__init__()
        self._value = RandomNormalBase(mean)
    
    def _getValue(self, item):
        return self._value.value


from util.random import RandomWeighted as RandomWeightedBase
class RandomWeighted(_Value):
    """
    A wrapper for <util.random.RandomWeighted>
    """
    def __init__(self, distribution):
        super().__init__()
        self._value = RandomWeightedBase(distribution)
    
    def _getValue(self, item):
        return self._value.value